<?php
require __DIR__ . '/vendor/autoload.php';
require_once 'google_client_helper.php'; // contains getGoogleClient()

use Google\Service\Sheets;
use Google\Service\Sheets\ClearValuesRequest;
use Google\Service\Sheets\ValueRange;

function generateDeansList(Google_Client $client, string $spreadsheetId, string $outputSheetName = "Dean's List", array $excludeSheets = []) {
    try {
        $service = new Sheets($client);
        error_log("=== Starting Dean's List generation ===");
        error_log("Spreadsheet ID: " . $spreadsheetId);

        // ✅ Ensure output sheet is excluded
        if (!in_array($outputSheetName, $excludeSheets)) {
            $excludeSheets[] = $outputSheetName;
        }

        // ✅ Get all sheet tabs except excluded
        $spreadsheet = $service->spreadsheets->get($spreadsheetId);
        $sheetsMeta = $spreadsheet->getSheets();
        $subjectSheets = [];
        foreach ($sheetsMeta as $smeta) {
            $title = $smeta->getProperties()->getTitle();
            if (in_array($title, $excludeSheets)) continue;
            $subjectSheets[] = $title;
        }

        error_log("Sheets found (excluding): " . implode(", ", $subjectSheets));

        $totalSubjects = count($subjectSheets);
        if ($totalSubjects === 0) {
            error_log("No subject sheets found.");
            throw new Exception("No subject sheets found.");
        }

        $students = [];

        // 🔹 READ EACH SUBJECT SHEET
        foreach ($subjectSheets as $sheetTitle) {
            $safeTitle = "'" . str_replace("'", "''", $sheetTitle) . "'";

            // 🔹 Get unit value from AB6
            $unitRange = "{$safeTitle}!AB6";
            $unitResp = $service->spreadsheets_values->get($spreadsheetId, $unitRange);
            $unitValues = $unitResp->getValues();
            $subjectUnit = isset($unitValues[0][0]) && is_numeric($unitValues[0][0]) ? floatval($unitValues[0][0]) : 0;
            error_log("Sheet: {$sheetTitle} | Units: {$subjectUnit}");

            if ($subjectUnit <= 0) {
                error_log("Skipping {$sheetTitle}: invalid or zero units");
                continue;
            }

            // 🔹 Read student IDs and grades
            $range = "{$safeTitle}!B12:AD1000";
            $resp = $service->spreadsheets_values->get($spreadsheetId, $range);
            $rows = $resp->getValues();

            error_log("Sheet {$sheetTitle}: rows fetched = " . count($rows));

            if (empty($rows)) continue;

            foreach ($rows as $i => $row) {
                $id = isset($row[0]) ? trim($row[0]) : '';       // Column B
                $finalRaw = isset($row[28]) ? trim($row[28]) : ''; // Column AD

                if ($id === '' || !is_numeric($finalRaw)) continue;

                $grade = floatval($finalRaw);
                if ($grade <= 0) continue;

                if (!isset($students[$id])) {
                    $students[$id] = [
                        'id' => $id,
                        'grades' => [],
                        'units' => [],
                    ];
                }

                $students[$id]['grades'][$sheetTitle] = $grade;
                $students[$id]['units'][$sheetTitle] = $subjectUnit;
            }
        }

        error_log("Total students found: " . count($students));

        // 🔹 COMPUTE GWA & QUALIFY
        $qualified = [];
        foreach ($students as $id => $info) {
            $gradesBySubject = $info['grades'];
            $unitsBySubject = $info['units'];

            $takenSubjects = count($gradesBySubject);
            if ($takenSubjects < $totalSubjects) {
                error_log("Student {$id} skipped: missing subjects ($takenSubjects / $totalSubjects)");
                continue;
            }

            $totalWeighted = 0;
            $totalUnits = 0;
            $hasLowGrade = false;

            foreach ($gradesBySubject as $subject => $grade) {
                $unit = $unitsBySubject[$subject] ?? 0;
                if ($grade < 80) {
                    $hasLowGrade = true;
                    error_log("Student {$id} skipped: low grade {$grade} in {$subject}");
                    break;
                }
                $totalWeighted += $grade * $unit;
                $totalUnits += $unit;
            }

            if ($hasLowGrade || $totalUnits == 0) continue;

            $gwaPrecise = $totalWeighted / $totalUnits;
            $gwa = round($gwaPrecise, 3);
            error_log("Student {$id} | GWA: {$gwa}");

            if ($gwa >= 90) {
                $qualified[] = ['id' => $id, 'gwa' => $gwa];
            }
        }

        error_log("Qualified students: " . count($qualified));

        // 🔹 SORT AND OUTPUT
        if (!empty($qualified)) {
            usort($qualified, function($a, $b) {
                return $b['gwa'] <=> $a['gwa'];
            });
        }

        $out = [["Rank", "ID#", "GWA"]];
        $rank = 1;
        foreach ($qualified as $q) {
            $out[] = [$rank++, $q['id'], $q['gwa']];
        }

        // 🔹 Write results
        $sheetExists = false;
        foreach ($spreadsheet->getSheets() as $sh) {
            if ($sh->getProperties()->getTitle() === $outputSheetName) {
                $sheetExists = true;
                break;
            }
        }

        if (!$sheetExists) {
            $addReq = new \Google\Service\Sheets\Request([
                'addSheet' => ['properties' => ['title' => $outputSheetName]]
            ]);
            $batch = new \Google\Service\Sheets\BatchUpdateSpreadsheetRequest([
                'requests' => [$addReq]
            ]);
            $service->spreadsheets->batchUpdate($spreadsheetId, $batch);
        }

        $safeOut = str_replace("'", "''", $outputSheetName);
        $rangeToClear = "'{$safeOut}'!A1:C1000";
        $rangeToWrite = "'{$safeOut}'!A1";

        $clearReq = new ClearValuesRequest();
        $service->spreadsheets_values->clear($spreadsheetId, $rangeToClear, $clearReq);

        $body = new ValueRange(['values' => $out]);
        $params = ['valueInputOption' => 'RAW'];
        $service->spreadsheets_values->update($spreadsheetId, $rangeToWrite, $body, $params);

        error_log("Dean's List generation completed. " . (count($out) - 1) . " students listed.");
        return [
            "status" => "success",
            "message" => "Dean's List generated successfully. " . (count($out) - 1) . " student(s) listed."
        ];

    } catch (Exception $e) {
        error_log("❌ Exception: " . $e->getMessage());
        return [
            "status" => "error",
            "message" => "❌ Error: " . $e->getMessage()
        ];
    }
}

/*
|-----------------------------------------------------------
|  🧩 AJAX ENTRY POINT
|-----------------------------------------------------------
*/
if (isset($_POST['spreadsheetId'])) {
    header('Content-Type: application/json');
    try {
        $client = getGoogleClient();
        $spreadsheetId = $_POST['spreadsheetId'];
        $result = generateDeansList(
            $client,
            $spreadsheetId,
            "Dean's List",
            ["Template(do not modify)", "DO NOT DELETE"]
        );
        echo json_encode($result);
    } catch (Exception $e) {
        error_log("AJAX ❌ Exception: " . $e->getMessage());
        echo json_encode(["status" => "error", "message" => $e->getMessage()]);
    }
    exit;
}
?>
