<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>CCS Grading Sheet</title>
<link rel="icon" href="assets/images/favicon.ico" type="image/x-icon">
<style>
  @page { size: A4; margin: 20mm; }
  @media print { body {-webkit-print-color-adjust: exact; color-adjust: exact;} .no-print{display:none;} }
  body { font-family: "Times New Roman", serif; margin:0 auto; width:210mm; max-width:100%; background:#fff; color:#000; font-size:14px; line-height:1.2; }
  .banner { text-align:center; margin-top:10px; }
  .banner img { max-width:100%; width:900px; height:auto; }
  h2 { text-align:center; margin:10px 0; text-transform:uppercase; font-weight:bold; font-size:20px; }

  .header-info { margin-bottom:10px; }
  .header-info table { width:100%; border-collapse:collapse; font-size:14px; }
  .header-info td { padding:4px 6px; vertical-align:top; }

  table.grades { width:100%; border-collapse:collapse; margin-top:10px; font-size:13.5px; }
  table.grades th, table.grades td { border:1px solid #000; padding:6px; text-align:center; }
  table.grades th { background-color:#d9d9d9; }
  table.grades th:nth-child(2), table.grades td:nth-child(2) { text-align:left; }

  .grading-system { margin-top:25px; font-size:13.5px; }
  .grading-system table { width:100%; border-collapse:collapse; }
  .grading-system td { border:none; padding:3px 8px; text-align:left; }
  .grading-system .notes { margin-top:10px; font-size:13.5px; }

  .signatures { margin-top:40px; font-size:14px; }
  .signatures table { width:100%; border-collapse:collapse; }
  .signatures td { padding:10px; vertical-align:top; }
  .signature-line { border-top:1px solid #000; width:250px; margin-top:30px; text-align:center; padding-top:4px; }

  .no-print { margin-top:15px; text-align:center; }
  button { padding:8px 16px; font-size:14px; cursor:pointer; }
</style>
</head>
<body>
<div class="printable-area">
<?php
include 'db.php';
include_once 'gs_grading_sheet.php';

if(isset($_GET['id'])){
    $id = intval($_GET['id']); // sanitize class ID

    // ===== Fetch class info =====
    $stmt = $conn->prepare("
        SELECT 
            c.CLASS_ID, c.SECTION_ID, c.FACULTY_ID, c.SUBJECT_ID, c.CTIME, c.CDAY,
            sctn.SCHOOL_YEAR, sctn.YEAR_SECTION, sctn.GSID, sctn.SEMESTER,
            s.SUBJECT_CODE, s.UNITS, s.DESCRIPTION, s.YEAR_LEVEL, s.PROGRAM_ID,
            u.FIRSTNAME, u.LASTNAME, u.DESIGNATION, u.EXPERTISE, p.PROGRAM
        FROM classes c
        JOIN subjects s ON c.SUBJECT_ID = s.SUBJECT_ID
        JOIN section sctn ON c.SECTION_ID = sctn.SECTION_ID
        JOIN user u ON c.FACULTY_ID = u.USER_ID
        JOIN program p ON s.PROGRAM_ID = p.PROGRAM_ID
        WHERE c.CLASS_ID = ?
    ");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $result = $stmt->get_result();
    $class = $result->fetch_assoc();

    if(!$class){
        echo "<p>No class found.</p>";
        exit;
    }

    // ===== Google Sheets Student Data =====
    $spreadsheetId = $class['GSID'];
    $sheetName = $class['SUBJECT_CODE'];
    $students = getStudentsTotalAverage($spreadsheetId, $sheetName);

    // ===== Helper functions =====
    require_once 'calc_grades.php';

    function formatName($name){
        $parts = explode(',', $name);
        $lastName = isset($parts[0]) ? strtoupper(trim($parts[0])) : '';
        $firstMiddle = '';
        if(isset($parts[1])){
            $words = explode(' ', trim($parts[1]));
            $words = array_map(function($w){ return ucfirst(strtolower($w)); }, $words);
            $firstMiddle = implode(' ', $words);
        }
        return $lastName . ($firstMiddle ? ', ' . $firstMiddle : '');
    }

    function getRemark($grade){
        $grade = trim($grade);
        if($grade === '' || floatval($grade)==0) return 'INC';
        elseif(floatval($grade)>=75) return 'Passed';
        else return 'Failed';
    }
?>

<!-- ===== Image Banner ===== -->
<div class="banner">
  <img src="assets/images/banner.jpg" alt="School Banner">
</div>

<h2>Grading Sheet</h2>

<?php if (is_string($students)) {
    // Display error message if sheet not found
    echo "<p style='color:red; text-align:center; font-weight:bold;'>{$students}</p>";
} else { 
    // Sort students alphabetically
    usort($students, function($a,$b){
        return strcmp($a['name'], $b['name']);
    });
?>

<!-- ===== Class Header Info ===== -->
<div class="header-info">
  <table>
    <tr>
      <td><b>COURSE NO.:</b> <?= htmlspecialchars($class['SUBJECT_CODE']) ?></td>
      <td><b>SEMESTER:</b> <?= htmlspecialchars($class['SEMESTER']) ?> Sem</td>
    </tr>
    <tr>
      <td><b>DESCRIPTION:</b> <?= htmlspecialchars($class['DESCRIPTION']) ?></td>
      <td><b>UNITS:</b> <?= number_format($class['UNITS'],1) ?></td>
    </tr>
    <tr>
      <td><b>ACADEMIC YEAR:</b> <?= htmlspecialchars($class['SCHOOL_YEAR']) ?></td>
      <td><b>TIME:</b> <?= htmlspecialchars($class['CTIME']) ?>&nbsp; <b>DAY:</b> <?= htmlspecialchars($class['CDAY']) ?></td>
    </tr>
  </table>
</div>

<!-- ===== Students Table ===== -->
<table class="grades">
  <tr>
    <th style="width:5%;">No.</th>
    <th style="width:35%;">Names in Alphabetical Order<br>(Surname, First Name, M.I.)</th>
    <th style="width:10%;">Sex</th>
    <th style="width:20%;">Course & Year</th>
    <th style="width:15%;">Final Grade</th>
    <th style="width:15%;">Remark/s</th>
  </tr>

<?php
$count = 0;
foreach($students as $student):
    $query = $conn->prepare("SELECT GENDER FROM student_info WHERE LRN_NO = ? LIMIT 1");
    $query->bind_param("s", $student['id']);
    $query->execute();
    $result = $query->get_result();
    $student_row = $result->fetch_assoc();
    $gender = isset($student_row['GENDER']) ? strtoupper($student_row['GENDER'][0]) : '-';
?>
<tr>
    <td><?= ++$count ?></td>
    <td><?= htmlspecialchars(formatName($student['name'])) ?>.</td>
    <td><?= $gender ?></td>
    <td><?= $class['PROGRAM'].' '.$class['YEAR_SECTION'][0] ?></td>
    <td><?php echo calc_grade($student['average']) ?></td>
    <td><?= getRemark($student['average']) ?></td>
</tr>
<?php endforeach; ?>
</table>
<?php } // end else ?>


<!-- ===== Add New Line Button ===== -->
<div class="no-print" style="text-align:left; margin-top:5px;">
  <button id="addLineBtn" title="Add a New Line to fix format when printing">+</button>
</div>
<div id="blank-lines-container"></div>
</div>
<!-- ===== Grading System ===== -->
<div class="grading-system">
  <h4>Grading System:</h4>
  <table>
    <tr>
      <td><b>1.00</b> – 99–100%</td>
      <td><b>1.40</b> – 94%</td>
      <td><b>1.80</b> – 89%</td>
      <td><b>2.25</b> – 84%</td>
      <td><b>2.70</b> – 79%</td>
    </tr>
    <tr>
      <td><b>1.10</b> – 98%</td>
      <td><b>1.50</b> – 93%</td>
      <td><b>1.90</b> – 88%</td>
      <td><b>2.30</b> – 83%</td>
      <td><b>2.75</b> – 78%</td>
    </tr>
    <tr>
      <td><b>1.20</b> – 97%</td>
      <td><b>1.60</b> – 92%</td>
      <td><b>2.00</b> – 87%</td>
      <td><b>2.40</b> – 82%</td>
      <td><b>2.80</b> – 77%</td>
    </tr>
    <tr>
      <td><b>1.25</b> – 96%</td>
      <td><b>1.70</b> – 91%</td>
      <td><b>2.10</b> – 86%</td>
      <td><b>2.50</b> – 81%</td>
      <td><b>2.90</b> – 76%</td>
    </tr>
    <tr>
      <td><b>1.30</b> – 95%</td>
      <td><b>1.75</b> – 90%</td>
      <td><b>2.20</b> – 85%</td>
      <td><b>2.60</b> – 80%</td>
      <td><b>3.00</b> – 75%</td>
    </tr>
  </table>
  <div class="notes">
    <p><b>4.00</b> – Conditional (for mid-term only)</p>
    <p><b>Inc.</b> – Incomplete with remarks (reasons)</p>
    <p><b>5.00</b> – Failed</p>
  </div>
</div>

<!-- ===== Signatures ===== -->
<div class="signatures">
  <p><b>Certified Correct:</b></p>
  <p class="signature-line"><b><?php echo $class['FIRSTNAME'].' '.$class['LASTNAME']?></b><br>Printed Name of Faculty and Signature</p>

  <table>
    <tr>
      <td>
        <b>Approved:</b><br><br>
        <div class="signature-line"><b>ALDARUHZ T. DARKIS</b><br>Dean</div>
      </td>
      <td>
        <b>Noted:</b><br><br>
        <div class="signature-line"><b>PROF. NELSON U. JULHAMID, PhD.</b><br>Vice President for Academic Affairs</div>
      </td>
    </tr>
  </table>
</div>

<div class="no-print">
  <button onclick="window.location.href='index.php'"> 🏠 Go to Home </button>
  <button onclick="window.print()">🖨️ Print Grade Sheet</button>
</div>

<?php } // end if isset id ?>
</div>
<script>
// Append a blank line inside the dedicated container to avoid moving the button
document.getElementById('addLineBtn').addEventListener('click', () => {
    const container = document.getElementById('blank-lines-container');
    const newLine = document.createElement('div');
    newLine.style.height = '20px'; // Adjust spacing for blank line
    container.appendChild(newLine);
});
</script>
</body>
</html>
