<?php
require __DIR__ . '/vendor/autoload.php';
require_once 'google_client_helper.php'; // contains getGoogleClient()

use Google\Service\Sheets;
use Google\Service\Sheets\ClearValuesRequest;
use Google\Service\Sheets\ValueRange;

function generateDeansList(Google_Client $client, string $spreadsheetId, string $outputSheetName = "Dean's List", array $excludeSheets = []) {
    try {
        $service = new Sheets($client);

        // ✅ Ensure output sheet is excluded
        if (!in_array($outputSheetName, $excludeSheets)) {
            $excludeSheets[] = $outputSheetName;
        }

        // ✅ Get all sheet tabs except excluded
        $spreadsheet = $service->spreadsheets->get($spreadsheetId);
        $sheetsMeta = $spreadsheet->getSheets();
        $subjectSheets = [];
        foreach ($sheetsMeta as $smeta) {
            $title = $smeta->getProperties()->getTitle();
            if (in_array($title, $excludeSheets)) continue;
            $subjectSheets[] = $title;
        }

        $totalSubjects = count($subjectSheets);
        if ($totalSubjects === 0) {
            throw new Exception("No subject sheets found.");
        }

        $students = [];

        /*
        |-----------------------------------------------------------
        | 1️⃣  READ ID (col B) and TOTAL AVERAGE (col AD) from each sheet
        |-----------------------------------------------------------
        */
        foreach ($subjectSheets as $sheetTitle) {
            $safeTitle = "'" . str_replace("'", "''", $sheetTitle) . "'";
            $range = $safeTitle . "!B12:AD1000"; // ID in B, Final in AD
            $resp = $service->spreadsheets_values->get($spreadsheetId, $range);
            $rows = $resp->getValues();

            if (empty($rows)) continue;
            //error_log("✅ Reading sheet: $sheetTitle, rows: " . count($rows));

            foreach ($rows as $i => $row) {
                $id = isset($row[0]) ? trim($row[0]) : ''; // Column B (ID)
                $finalRaw = isset($row[28]) ? trim($row[28]) : ''; // Column AD (Final)

                //error_log("Row $i -> ID: [$id], Final: [$finalRaw]");

                if ($id === '' || !is_numeric($finalRaw)) continue;

                $grade = floatval($finalRaw);
                if (!isset($students[$id])) {
                    $students[$id] = [
                        'id' => $id,
                        'grades' => []
                    ];
                }

                $students[$id]['grades'][$sheetTitle] = $grade;
            }
        }

        /*
        |-----------------------------------------------------------
        | 2️⃣  DETERMINE QUALIFIED STUDENTS
        |-----------------------------------------------------------
        */
        $qualified = [];
        foreach ($students as $id => $info) {
            $gradesBySubject = $info['grades'];
            $takenSubjects = count($gradesBySubject);

            //error_log("📘 Student $id -> subjects: $takenSubjects / $totalSubjects -> " . json_encode($gradesBySubject));

            if ($takenSubjects < $totalSubjects) {
                //error_log("⚠️ Skipping $id: incomplete subjects ($takenSubjects/$totalSubjects)");
                continue;
            }

            $validGrades = array_map('floatval', $gradesBySubject);

            if (min($validGrades) < 80) {
                //error_log("⚠️ Skipping $id: has grade below 80 (" . min($validGrades) . ")");
                continue;
            }
            
            $gwa = round(array_sum($validGrades) / $takenSubjects, 2);
            
            if ($gwa >= 90) { // Dean’s List condition
                $qualified[] = [
                    'id' => $id,
                    'gwa' => $gwa
                ];
                //error_log("✅ Qualified $id with GWA: $gwa (Dean’s Lister)");
            } else {
                //error_log("⚠️ Skipping $id: GWA below 90 ($gwa)");
            }

            //error_log("✅ Qualified $id with GWA: $gwa");
        }

        /*
        |-----------------------------------------------------------
        | 3️⃣  SORT AND OUTPUT
        |-----------------------------------------------------------
        */
        usort($qualified, function ($a, $b) {
            return $b['gwa'] <=> $a['gwa'];
        });

        $out = [["Rank", "ID#", "GWA"]];
        $rank = 1;
        foreach ($qualified as $q) {
            $out[] = [$rank++, $q['id'], $q['gwa']];
        }

        // ✅ Write results
        $safeOut = str_replace("'", "''", $outputSheetName);
        $sheetExists = false;
        foreach ($spreadsheet->getSheets() as $sh) {
            if ($sh->getProperties()->getTitle() === $outputSheetName) {
                $sheetExists = true;
                break;
            }
        }

        if (!$sheetExists) {
            $addReq = new \Google\Service\Sheets\Request([
                'addSheet' => ['properties' => ['title' => $outputSheetName]]
            ]);
            $batch = new \Google\Service\Sheets\BatchUpdateSpreadsheetRequest([
                'requests' => [$addReq]
            ]);
            $service->spreadsheets->batchUpdate($spreadsheetId, $batch);
        }

        $rangeToClear = "'{$safeOut}'!A1:C1000";
        $rangeToWrite = "'{$safeOut}'!A1";
        $clearReq = new ClearValuesRequest();
        $service->spreadsheets_values->clear($spreadsheetId, $rangeToClear, $clearReq);
        $body = new ValueRange(['values' => $out]);
        $params = ['valueInputOption' => 'RAW'];
        $service->spreadsheets_values->update($spreadsheetId, $rangeToWrite, $body, $params);

        $sheetUrl = "https://docs.google.com/spreadsheets/d/$spreadsheetId/edit";
        return [
            "status" => "success",
            "message" => "Dean's List generated successfully. " . (count($out) - 1) . " students listed."
        ];

    } catch (Exception $e) {
        return [
            "status" => "error",
            "message" => "❌ Error: " . $e->getMessage()
        ];
    }
}

/*
|-----------------------------------------------------------
|  🧩 AJAX ENTRY POINT
|-----------------------------------------------------------
*/
if (isset($_POST['spreadsheetId'])) {
    header('Content-Type: application/json');
    try {
        $client = getGoogleClient();
        $spreadsheetId = $_POST['spreadsheetId'];
        $result = generateDeansList(
            $client,
            $spreadsheetId,
            "Dean's List",
            ["Template(do not modify)", "DO NOT DELETE"]
        );
        echo json_encode($result);
    } catch (Exception $e) {
        echo json_encode(["status" => "error", "message" => $e->getMessage()]);
    }
    exit;
}
?>
