<?php
require_once __DIR__ . '/google_client_helper.php';

function getAllStudentsGrades($spreadsheetIds) {
    $client = getGoogleClient();
    $service = new Google_Service_Sheets($client);

    $students = []; // master array, keyed by student ID

    foreach ($spreadsheetIds as $spreadsheetId) {
        $spreadsheet = $service->spreadsheets->get($spreadsheetId);
        $sheets = $spreadsheet->getSheets();

        foreach ($sheets as $sheet) {
            $sheetName = $sheet->getProperties()->getTitle();
            $range = $sheetName . "!A:Z"; // ID in col A, Name in col B, Grade in col C

            $response = $service->spreadsheets_values->get($spreadsheetId, $range);
            $values = $response->getValues();

            if (empty($values)) continue;

            foreach ($values as $rowIndex => $row) {
                // skip header row
                if ($rowIndex === 0) continue;

                if (!isset($row[0]) || trim($row[0]) === '') continue;

                $studentId = trim($row[0]);
                $studentName = $row[1] ?? '';
                $grade = $row[2] ?? ''; // adjust if different structure

                // Initialize student if not exists
                if (!isset($students[$studentId])) {
                    $students[$studentId] = [
                        "ID" => $studentId,
                        "Name" => $studentName,
                        "Subjects" => []
                    ];
                }

                // Save grade under the subject name (sheet tab name)
                $students[$studentId]["Subjects"][$sheetName] = $grade;
            }
        }
    }

    return $students;
}

// Example usage
$spreadsheetIds = [
    "1cgkat6zcKM25nsRVlSQ3Ma2MLAmovxZJopK47NRI3tk",
    "1HsOVcIwu1NEMJT-x97cPSfqsOBum_zK0UYUZd-nCI9U"
];

$allGrades = getAllStudentsGrades($spreadsheetIds);

// Show result
echo "<pre>";
print_r($allGrades);
echo "</pre>";

// OPTIONAL: export to CSV file
$fp = fopen("all_students_grades.csv", "w");

// Collect all subject names dynamically
$allSubjects = [];
foreach ($allGrades as $student) {
    $allSubjects = array_merge($allSubjects, array_keys($student["Subjects"]));
}
$allSubjects = array_unique($allSubjects);

// Write header row: ID, Name, Subjects...
fputcsv($fp, array_merge(["ID", "Name"], $allSubjects));

// Write each student row
foreach ($allGrades as $student) {
    $row = [$student["ID"], $student["Name"]];
    foreach ($allSubjects as $subj) {
        $row[] = $student["Subjects"][$subj] ?? ''; // leave blank if missing
    }
    fputcsv($fp, $row);
}

fclose($fp);

echo "Exported to all_students_grades.csv\n";
