<?php
require __DIR__ . '/vendor/autoload.php';
require_once 'google_client_helper.php'; // contains getGoogleClient()

use Google\Service\Sheets;
use Google\Service\Sheets\ClearValuesRequest;
use Google\Service\Sheets\ValueRange;

/**
 * Generates the Dean's List for a given Google Spreadsheet.
 *
 * Customize column positions, passing grade, and output layout inside the marked sections.
 */
function generateDeansList(Google_Client $client, string $spreadsheetId, string $outputSheetName, array $excludeSheets = []) {
    try {
        $service = new Sheets($client);

        // ✅ Exclude output sheet from being processed as subject
        if (!in_array($outputSheetName, $excludeSheets)) {
            $excludeSheets[] = $outputSheetName;
        }

        /*
        |-----------------------------------------------------------
        | 1️⃣  DISCOVER SUBJECT SHEETS
        |-----------------------------------------------------------
        */
        $spreadsheet = $service->spreadsheets->get($spreadsheetId);
        $sheetsMeta = $spreadsheet->getSheets();
        $subjectSheets = [];
        foreach ($sheetsMeta as $smeta) {
            $title = $smeta->getProperties()->getTitle();
            if (in_array($title, $excludeSheets)) continue;
            $subjectSheets[] = $title;
        }

        $totalSubjects = count($subjectSheets);
        if ($totalSubjects === 0) {
            throw new Exception("No subject sheets found.");
        }

        /*
        |-----------------------------------------------------------
        | 2️⃣  COLLECT STUDENT NAMES AND GRADES
        |-----------------------------------------------------------
        */
        $students = [];
        foreach ($subjectSheets as $sheetTitle) {
            // Escaping sheet title in case it has spaces or special chars
            $safeTitle = "'" . str_replace("'", "''", $sheetTitle) . "'";
            $range = $safeTitle . "!A:B";  // adjust as needed
            $resp = $service->spreadsheets_values->get($spreadsheetId, $range);
            $rows = $resp->getValues();
            if (empty($rows)) continue;

            foreach ($rows as $row) {
                // Student name
                $rawName = isset($row[0]) ? trim($row[0]) : ''; // Column C
                if ($rawName === '') continue;

                $nameKey = mb_strtolower(preg_replace('/\s+/', ' ', $rawName));

                if (!isset($students[$nameKey])) {
                    $students[$nameKey] = [
                        'displayName' => $rawName,
                        'grades' => []
                    ];
                }

                // Student grade
                $gradeRaw = isset($row[1]) ? trim($row[1]) : ''; // Column AD
                if ($gradeRaw === '' || !is_numeric($gradeRaw)) continue;


                $num = floatval(preg_replace('/[^0-9.\-]/', '', $gradeRaw));
                $students[$nameKey]['grades'][$sheetTitle] = $num;
            }
        }

        /*
        |-----------------------------------------------------------
        | 3️⃣  DETERMINE QUALIFIED STUDENTS
        |-----------------------------------------------------------
        */
        $qualified = [];
        foreach ($students as $info) {
            $gradesBySubject = $info['grades'];

            if (count($gradesBySubject) !== $totalSubjects) continue; // must have all subjects
            if (min($gradesBySubject) < 80) continue; // disqualify if any grade < 80

            $gwa = array_sum($gradesBySubject) / $totalSubjects;

            $qualified[] = [
                'name' => $info['displayName'],
                'gwa' => round($gwa, 2)
            ];
        }

        /*
        |-----------------------------------------------------------
        | 4️⃣  SORT RESULTS
        |-----------------------------------------------------------
        */
        usort($qualified, function ($a, $b) {
            return $b['gwa'] <=> $a['gwa'];
        });

        /*
        |-----------------------------------------------------------
        | 5️⃣  PREPARE OUTPUT
        |-----------------------------------------------------------
        */
        $out = [["Rank", "Name", "GWA"]];
        $rank = 1;
        foreach ($qualified as $q) {
            $out[] = [$rank++, $q['name'], $q['gwa']];
        }

        /*
        |-----------------------------------------------------------
        | 6️⃣  WRITE RESULTS BACK TO GOOGLE SHEETS
        |-----------------------------------------------------------
        */
        $safeSheetName = str_replace("'", "''", $outputSheetName);

        // ✅ Check if sheet exists
        $spreadsheet = $service->spreadsheets->get($spreadsheetId);
        $sheetExists = false;
        foreach ($spreadsheet->getSheets() as $sheet) {
            if ($sheet->getProperties()->getTitle() === $outputSheetName) {
                $sheetExists = true;
                break;
            }
        }

        // ✅ Create if missing
        if (!$sheetExists) {
            $addSheetRequest = new Google\Service\Sheets\Request([
                'addSheet' => ['properties' => ['title' => $outputSheetName]]
            ]);

            $batchUpdateRequest = new Google\Service\Sheets\BatchUpdateSpreadsheetRequest([
                'requests' => [$addSheetRequest]
            ]);

            $service->spreadsheets->batchUpdate($spreadsheetId, $batchUpdateRequest);
        }

        // ✅ Define ranges safely
        $rangeToClear = "'{$safeSheetName}'!A3:C3";
        $rangeToWrite = "'{$safeSheetName}'!A3";

        // Clear old data
        $clearReq = new Google\Service\Sheets\ClearValuesRequest();
        $service->spreadsheets_values->clear($spreadsheetId, $rangeToClear, $clearReq);

        // Write new data
        $body = new Google\Service\Sheets\ValueRange(['values' => $out]);
        $params = ['valueInputOption' => 'RAW'];
        $service->spreadsheets_values->update($spreadsheetId, $rangeToWrite, $body, $params);

        // ✅ Return success with link
        $sheetUrl = "https://docs.google.com/spreadsheets/d/$spreadsheetId/edit#gid=0";
        return [
            "status" => "success",
            "message" => "✅ Dean's List generated successfully. <br><b>" . (count($out) - 1) . "</b> students listed.<br><a href='$sheetUrl' target='_blank'>Open Google Sheet</a>"
        ];

    } catch (Exception $e) {
        return [
            "status" => "error",
            "message" => "❌ Error: " . $e->getMessage()
        ];
    }
}

/*
|-----------------------------------------------------------
|  🧩 AJAX ENTRY POINT
|-----------------------------------------------------------
*/

    try {
        $client = getGoogleClient();
        $spreadsheetId = "1EdJwilROsjdZ8XRYzscCh3lVP0oYwKah83EGDxsDZiY";
        $result = generateDeansList(
    $client,
    $spreadsheetId,
    "Dean's List",                 // Output sheet name
    ["Sheet1"]    // Sheets to exclude
);
        echo json_encode($result);
    } catch (Exception $e) {
        echo json_encode(["status" => "error", "message" => $e->getMessage()]);
    }


?>